// author: craig cox
//         rea computing, inc.
//         c.airspeed_applet@reacomp.com

import java.awt.*;

// a text field that accepts only doubles.  it can be initialized to accept
// only non-negative doubles.

public class DoubleTextField extends TextField
{
    private boolean m_bAllowNegative = true;
    private int     m_nSelectionEnd = 0;
    private int     m_nSelectionStart = 0;
    private String  m_sText = "";

    // create a text field that accepts only doubles
    
    public DoubleTextField( )
    {
        super( 9 );
    }
    
    public DoubleTextField( int cols )
    {
        super( cols );
    }
    
    // specify whether or not the text field should accept negative doubles.
    // if this method is not called, the text field will accept negative
    // doubles.
    
    public void allowNegative( boolean myAllowNegative )
    {
        m_bAllowNegative = myAllowNegative;
    }

    // event handler

    synchronized public boolean handleEvent( Event evt )
    {
        int     nSelectionEnd;
        int     nSelectionStart;
        String  sText;
        
        sText = this.getText( );
        nSelectionStart = this.getSelectionStart( );
        nSelectionEnd = this.getSelectionEnd( );

        switch ( evt.id )
        {
            case Event.KEY_PRESS:
            {
                if ( this.isAcceptablePartialEntry( sText ) )
                {
                    m_sText = sText;
                    m_nSelectionStart = nSelectionStart;
                    m_nSelectionEnd = nSelectionEnd;
                }

                break;
            }

            case Event.KEY_RELEASE:
            {
                if ( this.isAcceptablePartialEntry( sText ) )
                {
                    m_sText = sText;
                    m_nSelectionStart = nSelectionStart;
                    m_nSelectionEnd = nSelectionEnd;
                }
                else
                {
                    this.setText( m_sText );
                    this.select( m_nSelectionStart, m_nSelectionEnd );
                }

                break;
            }

            case Event.LOST_FOCUS:
            {
                if ( !sText.equals( "" ) )
                {
                    try
                    {
                        this.doubleValue( sText );
                        m_sText = sText;
                    }
                    catch ( NumberFormatException e )
                    {
                        this.requestFocus( );
                    }
                }

                break;
            }
        }

        return super.handleEvent( evt );
    }

    // returns true if the text in the text field evaluates to a valid double

    public boolean hasValidValue( )
    {
        return this.isValidValue( this.getText( ) );
    }

    // returns true if there are no characters in the text field

    public boolean isEmpty( )
    {
        return this.getText( ).equals( "" );
    }

    // returns the double value of the text in the text field

    public double doubleValue( )
    {
        return this.doubleValue( this.getText( ) );
    }

    // creates a string representation of the object

    public String toString( )
    {
        return this.toString( this.getText( ) );
    }

    // returns true if the entered text could be part of a valid entry

    private boolean isAcceptablePartialEntry( String sText )
    {
        if ( this.isValidValue( sText ) || sText.equals( "" )
            || sText.equals( "." ) )
        {
            return true;
        }

        if ( m_bAllowNegative && ( sText.equals( "-" )
            || sText.equals( "-." ) ) )
        {
            return true;
        }

        return false;
    }

    // returns true if the passed string evaluates to a valid double.  if the
    // text field does not allow negatives, the existence of a minus sign,
    // even for -0, makes the string invalid.  the empty string is invalid.
    // the string "-" is invalid.

    private boolean isValidValue( String sText )
    {
        Double Value;

        if ( sText == null || sText.length( ) == 0 )
        {
            return false;
        }

        try
        {
            Value = new Double( sText );
        }
        catch ( NumberFormatException e )
        {
            return false;
        }

        if ( !m_bAllowNegative && sText.indexOf( "-" ) != -1 )
        {
            return false;
        }

        return true;
    }

    // returns the double value of the passed string
    
    private double doubleValue( String sText )
    {
        if ( !this.isValidValue( sText ) )
        {
            throw new NumberFormatException( this.toString( sText ) );
        }

        return new Double( sText ).doubleValue( );
    }

    // creates a string representation of the object, allowing the entered
    // text to be passed instead of gotten from the object

    private String toString( String sText)
    {
        return new String( "Class: " + this.getClass( ).getName( )
            + "; Allow Negative: " + m_bAllowNegative + "; Text: " + sText );
    }
}
