// author: craig cox
//         rea computing, inc.
//         c.airspeed_applet@reacomp.com

import java.applet.Applet;
import java.awt.*;

// this class determines the true airspeed of an aircraft using groundspeeds
// obtained by flying on certain headings

public class TrueAirspeedApplet extends Applet
{
    private Button                     m_btnCompute;
    private Checkbox                   m_cbxShowIterations;
    private GroundtrackCanvas          m_cnvGroundtrack;
    private Label                      m_lblTrueAirspeed;
    private Label                      m_lblWind;
    private DoubleTextField            m_dtfEastGroundspeed;
    private DoubleTextField            m_dtfNorthGroundspeed;
    private DoubleTextField            m_dtfSouthGroundspeed;
    private DoubleTextField            m_dtfWestGroundspeed;
    private String                     m_sDirectionOrder = new String( );
    private TrueAirspeedWindEstimate   m_estimate =
                                           new TrueAirspeedWindEstimate( );

    // override the paint method to draw a border around the applet

    public void paint( Graphics g )
    {
        g.drawRect( 0, 0, this.size( ).width - 1, this.size( ).height - 1 );
        super.paint( g );
    }
    
    // override the init method to lay out the components in the applet

    public void init( )
    {
        GridBagLayout       gridbag = new GridBagLayout( );
        GridBagConstraints  c = new GridBagConstraints( );
        
        this.resize( 400, 220 );

        this.setLayout( gridbag );

        m_dtfNorthGroundspeed = new DoubleTextField( 5 );
        m_dtfNorthGroundspeed.allowNegative( false );
        c.gridx = 1;
        c.gridy = 0;
        c.gridwidth = 1;
        c.gridheight = 1;
        c.fill = GridBagConstraints.NONE;
        c.anchor = GridBagConstraints.SOUTH;
        gridbag.setConstraints( m_dtfNorthGroundspeed, c );
        this.add( m_dtfNorthGroundspeed );

        m_dtfWestGroundspeed = new DoubleTextField( 5 );
        m_dtfWestGroundspeed.allowNegative( false );
        c.gridx = 0;
        c.gridy = 1;
        c.gridwidth = 1;
        c.gridheight = 1;
        c.fill = GridBagConstraints.NONE;
        c.anchor = GridBagConstraints.EAST;
        gridbag.setConstraints( m_dtfWestGroundspeed, c );
        this.add( m_dtfWestGroundspeed );

        Canvas c1 = new GroundspeedCanvas( this );
        c.gridx = 1;
        c.gridy = 1;
        c.gridwidth = 1;
        c.gridheight = 1;
        c.fill = GridBagConstraints.NONE;
        c.anchor = GridBagConstraints.CENTER;
        gridbag.setConstraints( c1, c );
        this.add( c1 );

        m_dtfEastGroundspeed = new DoubleTextField( 5 );
        m_dtfEastGroundspeed.allowNegative( false );
        c.gridx = 2;
        c.gridy = 1;
        c.gridwidth = 1;
        c.gridheight = 1;
        c.fill = GridBagConstraints.NONE;
        c.anchor = GridBagConstraints.WEST;
        gridbag.setConstraints( m_dtfEastGroundspeed, c );
        this.add( m_dtfEastGroundspeed );

        m_dtfSouthGroundspeed = new DoubleTextField( 5 );
        m_dtfSouthGroundspeed.allowNegative( false );
        c.gridx = 1;
        c.gridy = 2;
        c.gridwidth = 1;
        c.gridheight = 1;
        c.fill = GridBagConstraints.NONE;
        c.anchor = GridBagConstraints.NORTH;
        gridbag.setConstraints( m_dtfSouthGroundspeed, c );
        this.add( m_dtfSouthGroundspeed );

        Label l1 = new Label( "Enter groundspeeds above", Label.CENTER );
        c.gridx = 0;
        c.gridy = 3;
        c.gridwidth = 3;
        c.gridheight = 1;
        c.fill = GridBagConstraints.NONE;
        c.anchor = GridBagConstraints.CENTER;
        gridbag.setConstraints( l1, c );
        this.add( l1 );

        m_btnCompute = new Button("Compute");
        m_btnCompute.disable( );
        c.gridx = 0;
        c.gridy = 4;
        c.gridwidth = 3;
        c.gridheight = GridBagConstraints.RELATIVE;
        c.fill = GridBagConstraints.NONE;
        c.anchor = GridBagConstraints.CENTER;
        gridbag.setConstraints( m_btnCompute, c );
        this.add( m_btnCompute );

        m_cbxShowIterations = new Checkbox( "Show iterations", null, true );
        c.gridx = 0;
        c.gridy = 5;
        c.gridwidth = 3;
        c.gridheight = GridBagConstraints.REMAINDER;
        c.fill = GridBagConstraints.NONE;
        c.anchor = GridBagConstraints.CENTER;
        gridbag.setConstraints( m_cbxShowIterations, c );
        this.add( m_cbxShowIterations );

        m_cnvGroundtrack = new GroundtrackCanvas( this );
        c.gridx = 4;
        c.gridy = 0;
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.gridheight = 3;
        c.fill = GridBagConstraints.NONE;
        c.anchor = GridBagConstraints.CENTER;
        gridbag.setConstraints( m_cnvGroundtrack, c );
        this.add( m_cnvGroundtrack );

        Label l2 = new Label( "Groundtrack displayed above", Label.CENTER );
        c.gridx = 4;
        c.gridy = 3;
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.gridheight = 1;
        c.fill = GridBagConstraints.NONE;
        c.anchor = GridBagConstraints.CENTER;
        gridbag.setConstraints( l2, c );
        this.add( l2 );

        Label l3 = new Label( "True airspeed: ", Label.RIGHT );
        c.gridx = 4;
        c.gridy = 4;
        c.gridwidth = GridBagConstraints.RELATIVE;
        c.gridheight = GridBagConstraints.RELATIVE;
        c.fill = GridBagConstraints.NONE;
        c.anchor = GridBagConstraints.EAST;
        gridbag.setConstraints( l3, c );
        this.add( l3 );

        m_lblTrueAirspeed = new Label( "            ", Label.LEFT );
        c.gridx = 5;
        c.gridy = 4;
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.gridheight = GridBagConstraints.RELATIVE;
        c.fill = GridBagConstraints.NONE;
        c.anchor = GridBagConstraints.WEST;
        gridbag.setConstraints( m_lblTrueAirspeed, c );
        this.add( m_lblTrueAirspeed );

        Label l4 = new Label( "Wind: ", Label.RIGHT );
        c.gridx = 4;
        c.gridy = 5;
        c.gridwidth = GridBagConstraints.RELATIVE;
        c.gridheight = GridBagConstraints.REMAINDER;
        c.fill = GridBagConstraints.NONE;
        c.anchor = GridBagConstraints.EAST;
        gridbag.setConstraints( l4, c );
        this.add( l4 );

        m_lblWind = new Label( "                        ", Label.LEFT );
        c.gridx = 5;
        c.gridy = 5;
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.gridheight = GridBagConstraints.REMAINDER;
        c.fill = GridBagConstraints.NONE;
        c.anchor = GridBagConstraints.WEST;
        gridbag.setConstraints( m_lblWind, c );
        this.add( m_lblWind );
    }

    // if focus goes to or a key is released in a DoubleTextField, call a
    // function to process the value

    public boolean handleEvent( Event evt )
    {
        if ( evt.target instanceof DoubleTextField )
        {
            if ( evt.id == Event.GOT_FOCUS || evt.id == Event.KEY_RELEASE )
            {
                this.addDirection( ( DoubleTextField ) evt.target );
            }
        }
        return super.handleEvent( evt );
    }

    // if the compute button is pressed, disable appropriate components,
    // compute the wind and airspeed, and re-enable the components

    public boolean action( Event evt, Object obj )
    {
        if ( evt.target == m_btnCompute )
        {
            m_dtfNorthGroundspeed.disable( );
            m_dtfEastGroundspeed.disable( );
            m_dtfSouthGroundspeed.disable( );
            m_dtfWestGroundspeed.disable( );
            m_btnCompute.disable( );
            m_cbxShowIterations.disable( );

            this.compute( );
            
            m_dtfNorthGroundspeed.enable( );
            m_dtfEastGroundspeed.enable( );
            m_dtfSouthGroundspeed.enable( );
            m_dtfWestGroundspeed.enable( );
            m_btnCompute.enable( );
            m_cbxShowIterations.enable( );
        }
        
        return super.action( evt, obj );
    }

    // process a groundspeed entered in an DoubleTextField

    private void addDirection( DoubleTextField dtf )
    {
        char    cDirection;
        double  dSpeed;
        int     nIndex;
        int     nLength;
    
        // get the direction of the groundspeed

        if ( dtf == m_dtfNorthGroundspeed )
        {
            cDirection = 'N';
        }
        else if ( dtf == m_dtfWestGroundspeed )
        {
            cDirection = 'W';
        }
        else if ( dtf == m_dtfEastGroundspeed )
        {
            cDirection = 'E';
        }
        else
        {
            cDirection = 'S';
        }
        
        // append (or move) the direction to the end of the direction order
        // string

        nLength = m_sDirectionOrder.length( );
        nIndex = m_sDirectionOrder.indexOf( cDirection );
        
        // remove the direction from the direction order string if it is
        // already there
        
        if ( nIndex != -1 )
        {
            if ( nIndex == 0 )
            {
                m_sDirectionOrder = m_sDirectionOrder.substring( 1 );
            }
            else if ( nIndex == nLength - 1 )
            {
                m_sDirectionOrder =
                    m_sDirectionOrder.substring( 0, nLength - 1 );
            }
            else
            {
                m_sDirectionOrder = m_sDirectionOrder.substring( 0, nIndex )
                    + m_sDirectionOrder.substring( nIndex + 1, nLength );
            }
        }
        
        // if the groundspeed is valid, add the direction to the end of the
        // direction order string

        if ( dtf.hasValidValue( ) )
        {
            m_sDirectionOrder = m_sDirectionOrder + cDirection;
        }

        // get the value entered, or -1 if the value is invalid

        try
        {
            dSpeed = dtf.doubleValue( );
        }
        catch ( NumberFormatException e )
        {
            dSpeed = -1.0;
        }
        
        // set the groundspeed in the current estimate of wind and airspeed

        m_estimate.setGroundspeed( cDirection, dSpeed );

        // remove any displays for the wind or airspeed

        m_lblTrueAirspeed.setText( "" );
        m_lblWind.setText( "" );
        
        // set the paint mode of the groundtrack canvas to ignore wind and
        // repaint it
        
        m_cnvGroundtrack.setPaintMode( false );
        m_cnvGroundtrack.repaint( );
        
        // the compute button is disabled if less than three groundspeeds have
        // been entered

        if ( m_sDirectionOrder.length( ) < 3 )
        {
            m_btnCompute.disable( );
        }
        else
        {
            m_btnCompute.enable( );
        }
    }
    
    // get the current extimate for wind and airspeed

    public TrueAirspeedWindEstimate getEstimate( )
    {
        return m_estimate;
    }
    
    // get the direction order string

    public String getDirectionOrder( )
    {
        return m_sDirectionOrder;
    }
    
    // compute the actual wind and aircraft airspeed

    private void compute( )
    {
        // set the paint mode of the groundtrack canvas to take into account
        // wind

        m_cnvGroundtrack.setPaintMode( true );
        
        // call the update function of the estimate of the wind and airspeed.
        // keep calling it until it returns true, meaning the correct wind and
        // airspeed have been determined.

        while ( !m_estimate.update( ) )
        {
            // if the user wants to see iterations, update the wind display,
            // airspeed display, and groundtrack for every successive guess

            if ( m_cbxShowIterations.getState( ) )
            {
                m_lblTrueAirspeed.setText( ""
                    + ( int ) Math.round( m_estimate.getTrueAirspeed( ) ) );
                m_lblWind.setText( m_estimate.getWind( ) );
                m_cnvGroundtrack.update( m_cnvGroundtrack.getGraphics( ) );
                m_cnvGroundtrack.paint( m_cnvGroundtrack.getGraphics( ) );
            
                // to prevent the updates from occurring so fast that the user
                // cannot see them, sleep for a tenth of a second

                try
                {
                    Thread.sleep( 100 );
                }
                catch ( InterruptedException e )
                {
                }
            }
        }
        
        // display the final (actual) wind, airspeed, and groundtrack

        m_lblTrueAirspeed.setText( ""
            + ( int ) Math.round( m_estimate.getTrueAirspeed( ) ) );
        m_lblWind.setText( m_estimate.getWind( ) );
        m_cnvGroundtrack.repaint( );
    }

    // this applet displays best at 400x220

    public Dimension minimumSize( )
    {
        return new Dimension( 400, 220 );
    }

    public Dimension preferredSize( )
    {
        return this.minimumSize( );
    }
}
