// author: craig cox
//         rea computing, inc.
//         c.airspeed_applet@reacomp.com

// this class encapsulates the data for an estimate of the true wind and
// airspeed and contains a method to get a better estimate

public class TrueAirspeedWindEstimate
{
    private double m_dEastGroundspeed = -1.0;
    private double m_dNorthGroundspeed = -1.0;
    private double m_dSouthGroundspeed = -1.0;
    private double m_dSouthWind = 0.0;
    private double m_dTrueAirspeed = 0.0;
    private double m_dWestGroundspeed = -1.0;
    private double m_dWestWind = 0.0;

    // return the wind as a string in the format wind_direction/wind_speed
    
    public String getWind( )
    {
        int     nDirection;
        int     nSpeed;
        String  sReturn;
    
        // determine the wind direction and speed as integers

        nDirection = ( int ) Math.round(
            ( Math.atan2( -m_dWestWind, -m_dSouthWind ) * 180 / Math.PI ) );
        nDirection = ( nDirection < 0 ) ? nDirection + 360 : nDirection;
        nSpeed = ( int ) Math.round( (
            Math.sqrt( m_dSouthWind * m_dSouthWind +
            m_dWestWind * m_dWestWind ) ) );

        // convert the integer wind direction and speed to a string and return
        // it

        sReturn = "00" + nDirection;
        sReturn = sReturn.substring( sReturn.length( ) - 3 ) + "/" + nSpeed;
        
        return sReturn;
    }
    
    // update the current estimates for wind and airspeed.  return true if the
    // update is complete, meaning the original estimate was already optimal
    // and could not be improved. return false if the update is not complete,
    // meaning the new estimate is different (and better) than the original.

    public boolean update( )
    {
        double  dBestError = Double.MAX_VALUE;
        double  dBestSouthWind = 0.0;
        double  dBestTrueAirspeed = 0.0;
        double  dBestWestWind = 0.0;
        double  dExpectedEastGroundspeed;
        double  dExpectedGroundspeed;
        double  dExpectedNorthGroundspeed;
        double  dTestError;
        double  dTestSouthWind;
        double  dTestTrueAirspeed;
        double  dTestWestWind;

        // make 27 attempts at finding a better solution.  vary the current
        // south wind by 1 or 0, vary the current west wind by 1 or 0, and
        // vary the current airspeed by 1 or 0.

        for ( dTestSouthWind = m_dSouthWind - 1.0;
              dTestSouthWind < m_dSouthWind + 2.0;
              dTestSouthWind++ )
        {
            for ( dTestWestWind = m_dWestWind - 1.0;
                  dTestWestWind < m_dWestWind + 2.0;
                  dTestWestWind++ )
            {
                for ( dTestTrueAirspeed = m_dTrueAirspeed - 1.0;
                      dTestTrueAirspeed < m_dTrueAirspeed + 2.0;
                      dTestTrueAirspeed++ )
                {
                    // accumulate an error which is the square of the
                    // difference between what we would have expected to see
                    // as a groundspeed on this estimate and what we actually
                    // saw

                    dTestError = 0.0;

                    if ( m_dNorthGroundspeed != -1 )
                    {
                        dExpectedNorthGroundspeed =
                            dTestTrueAirspeed + dTestSouthWind;
                        dExpectedEastGroundspeed = dTestWestWind;
                        dExpectedGroundspeed = Math.sqrt( 
                            ( dExpectedNorthGroundspeed
                            * dExpectedNorthGroundspeed ) + 
                            ( dExpectedEastGroundspeed
                            * dExpectedEastGroundspeed ) );
                        dTestError +=
                            Math.pow( m_dNorthGroundspeed
                            - dExpectedGroundspeed, 2.0 );
                    }

                    if ( m_dEastGroundspeed != -1 )
                    {
                        dExpectedNorthGroundspeed = dTestSouthWind;
                        dExpectedEastGroundspeed =
                            dTestTrueAirspeed + dTestWestWind;
                        dExpectedGroundspeed = Math.sqrt( 
                            ( dExpectedNorthGroundspeed
                            * dExpectedNorthGroundspeed ) + 
                            ( dExpectedEastGroundspeed *
                            dExpectedEastGroundspeed ) );
                        dTestError +=
                            Math.pow( m_dEastGroundspeed
                            - dExpectedGroundspeed, 2.0 );
                    }

                    if ( m_dSouthGroundspeed != -1 )
                    {
                        dExpectedNorthGroundspeed =
                            -dTestTrueAirspeed + dTestSouthWind;
                        dExpectedEastGroundspeed = dTestWestWind;
                        dExpectedGroundspeed = Math.sqrt( 
                            ( dExpectedNorthGroundspeed
                            * dExpectedNorthGroundspeed ) + 
                            ( dExpectedEastGroundspeed
                            * dExpectedEastGroundspeed ) );
                        dTestError +=
                            Math.pow( m_dSouthGroundspeed
                            - dExpectedGroundspeed, 2.0 );
                    }

                    if ( m_dWestGroundspeed != -1 )
                    {
                        dExpectedNorthGroundspeed = dTestSouthWind;
                        dExpectedEastGroundspeed =
                            -dTestTrueAirspeed + dTestWestWind;
                        dExpectedGroundspeed = Math.sqrt( 
                            ( dExpectedNorthGroundspeed
                            * dExpectedNorthGroundspeed ) + 
                            ( dExpectedEastGroundspeed
                            * dExpectedEastGroundspeed ) );
                        dTestError +=
                            Math.pow( m_dWestGroundspeed
                            - dExpectedGroundspeed, 2.0 );
                    }

                    // update the best wind and airspeed if the error for the
                    // new estimate is lower
                    
                    if ( ( dTestError < dBestError ) ||
                         ( dTestError == dBestError &&
                           dTestTrueAirspeed == m_dTrueAirspeed &&
                           dTestSouthWind == m_dSouthWind &&
                           dTestWestWind == m_dSouthWind ) )
                    {
                        dBestError = dTestError;
                        dBestTrueAirspeed = dTestTrueAirspeed;
                        dBestSouthWind = dTestSouthWind;
                        dBestWestWind = dTestWestWind;
                    }
                }
            }
        }
   
        // determine if the old estimate is still the best estimate

        if ( dBestTrueAirspeed == m_dTrueAirspeed &&
             dBestSouthWind == m_dSouthWind &&
             dBestWestWind == m_dWestWind )
        {
            // the old estimate is still the best estimate, so the best answer
            // has been found, so the update is complete, so return true
            
            return true;
        }
        else
        {
            // the new estimate is better than the old, so the best answer has
            // not yet been found (at least we do not know that yet), so the
            // update is not complete, so save the new best wind and airspeed
            // and return false

            m_dTrueAirspeed = dBestTrueAirspeed;
            m_dSouthWind = dBestSouthWind;
            m_dWestWind = dBestWestWind;
            
            return false;
        }
    }
    
    // set the groundspeed in a certain direction

    public void setGroundspeed( char cDirection, double dGroundspeed )
    {
        int dSpeed = 0;
        int nSpeedCount = 0;
    
        // set the instance variables containing the groundspeeds in certain
        // directions. a speed of -1.0 indicates that a groundspeed is not
        // available.

        switch( cDirection )
        {
            case 'N':
            {
                m_dNorthGroundspeed = dGroundspeed;
                break;
            }
            
            case 'E':
            {
                m_dEastGroundspeed = dGroundspeed;
                break;
            }
            
            case 'S':
            {
                m_dSouthGroundspeed = dGroundspeed;
                break;
            }
            
            case 'W':
            {
                m_dWestGroundspeed = dGroundspeed;
                break;
            }
        }
        
        // use the north and south groundspeeds to estimate the south wind

        if ( m_dNorthGroundspeed != -1 && m_dSouthGroundspeed != -1 )
        {
            m_dSouthWind = ( m_dNorthGroundspeed - m_dSouthGroundspeed ) / 2;
        }
        else
        {
            m_dSouthWind = 0.0;
        }
        
        // use the east and west groundspeeds to estimate the west wind

        if ( m_dEastGroundspeed != -1 && m_dWestGroundspeed != -1 )
        {
            m_dWestWind = ( m_dEastGroundspeed - m_dWestGroundspeed ) / 2;
        }
        else
        {
            m_dWestWind = 0.0;
        }
        
        // use all groundspeeds to estimate the aircraft true airspeed

        if ( m_dNorthGroundspeed != -1 )
        {
            dSpeed += m_dNorthGroundspeed;
            nSpeedCount++;
        }
        
        if ( m_dEastGroundspeed != -1 )
        {
            dSpeed += m_dEastGroundspeed;
            nSpeedCount++;
        }
        
        if ( m_dSouthGroundspeed != -1 )
        {
            dSpeed += m_dSouthGroundspeed;
            nSpeedCount++;
        }
        
        if ( m_dWestGroundspeed != -1 )
        {
            dSpeed += m_dWestGroundspeed;
            nSpeedCount++;
        }
        
        if ( nSpeedCount > 0 )
        {
            m_dTrueAirspeed = dSpeed / nSpeedCount;
        }
        else
        {
            m_dTrueAirspeed = 0.0;
        }
    }

    // get the groundspeed in a certain direction

    public double getGroundspeed( char cDirection )
    {
        double dSpeed = -1.0;
        
        switch( cDirection )
        {
            case 'N':
            {
                dSpeed = m_dNorthGroundspeed;
                break;
            }
            
            case 'E':
            {
                dSpeed = m_dEastGroundspeed;
                break;
            }
            
            case 'S':
            {
                dSpeed = m_dSouthGroundspeed;
                break;
            }
            
            case 'W':
            {
                dSpeed = m_dWestGroundspeed;
                break;
            }
        }
        
        return dSpeed;
    }
    
    // get the current estimate for true airspeed

    public double getTrueAirspeed( )
    {
        return m_dTrueAirspeed;
    }
    
    // get the current estimate for wind in a certain direction

    public double getWind( char cDirection )
    {
        double dWind = -1.0;
        
        switch ( cDirection )
        {
            case 'S':
            {
                dWind = m_dSouthWind;
                break;
            }
            
            case 'W':
            {
                dWind = m_dWestWind;
                break;
            }
        }
        
        return dWind;
    }
}
